const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ban')
        .setDescription('Oyuncuyu Discord ID ile banlar')
        .addStringOption(option =>
            option.setName('discordid')
                .setDescription('Oyuncunun Discord ID\'si')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('sebep')
                .setDescription('Ban sebebi')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('sure')
                .setDescription('Ban süresi')
                .setRequired(true)
                .addChoices(
                    { name: '1 Gün', value: '1 Day' },
                    { name: '3 Gün', value: '3 Day' },
                    { name: '7 Gün', value: '7 Day' },
                    { name: '1 Ay', value: '1 Month' },
                    { name: 'Kalıcı', value: 'Permanent' }
                )),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'ban')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const discordId = interaction.options.getString('discordid');
            const sebep = interaction.options.getString('sebep');
            const sure = interaction.options.getString('sure');
            const bannedBy = interaction.user.tag;

            // API isteği için body hazırla
            const requestBody = {
                discordId: discordId,
                banReason: sebep,
                banDuration: sure,
                bannedBy: bannedBy
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/ban`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const result = response.data.result;
                const successEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('🚫 Oyuncu Banlandı')
                    .setDescription(`Discord ID **${discordId}** başarıyla banlandı!`)
                    .addFields(
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '⏱️ Ban Süresi',
                            value: sure,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: sebep,
                            inline: false
                        }
                    )
                    .setFooter({ text: `RoveBot's • Ban Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'ban', {
                    color: 0xff0000,
                    title: '🚫 Ban Logu',
                    description: `Bir oyuncu banlandı`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '⏱️ Ban Süresi',
                            value: sure,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: sebep,
                            inline: false
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Oyuncu banlanırken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'ban', {
                    color: 0xff0000,
                    title: '❌ Ban Hatası',
                    description: `Ban işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Ban komutunda hata:', error);
            
            let errorMessage = 'Oyuncu banlanırken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'ban', {
                color: 0xff0000,
                title: '❌ Ban Sistem Hatası',
                description: `Ban işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

