const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('envanter-goruntule')
        .setDescription('Oyuncunun envanterini görüntüler')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'envantergoruntule')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/inventory`;
            const response = await axios.get(apiUrl, {
                params: {
                    playerId: oyuncuId
                },
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`
                }
            });

            if (response.data.success) {
                const inventory = response.data.inventory || [];
                
                // Envanter boş mu kontrol et
                if (inventory.length === 0) {
                    const emptyEmbed = new EmbedBuilder()
                        .setColor(0xffa500)
                        .setTitle('📦 Envanter Görüntüleme')
                        .setDescription(`Oyuncu ID **${oyuncuId}**'nin envanteri boş!`)
                        .addFields({
                            name: '👤 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        })
                        .setFooter({ text: `RoveBot's • Envanter Sistemi` })
                        .setTimestamp();

                    await interaction.editReply({ embeds: [emptyEmbed] });

                    // Log gönder
                    await sendLog(interaction.client, interaction.guild.id, 'envantergoruntule', {
                        color: 0xffa500,
                        title: '📦 Envanter Görüntüleme Logu',
                        description: `Bir oyuncunun envanteri görüntülendi (Boş)`,
                        user: interaction.user,
                        fields: [
                            {
                                name: '👤 Yetkili',
                                value: `${interaction.user} (${interaction.user.tag})`,
                                inline: true
                            },
                            {
                                name: '🎮 Oyuncu ID',
                                value: oyuncuId.toString(),
                                inline: true
                            },
                            {
                                name: '📦 Durum',
                                value: 'Envanter boş',
                                inline: true
                            }
                        ]
                    });
                    return;
                }

                // Envanteri table formatında göster
                let tableContent = '```\n';
                tableContent += 'Item Adı'.padEnd(25) + '| Miktar\n';
                tableContent += '-'.repeat(25) + '|' + '-'.repeat(10) + '\n';
                
                inventory.forEach(item => {
                    const itemName = (item.label || item.name || 'Bilinmeyen').substring(0, 24);
                    const count = `x${item.count || 1}`;
                    tableContent += itemName.padEnd(25) + '| ' + count + '\n';
                });
                
                tableContent += '```';

                // Embed oluştur
                const embed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('📦 Envanter Görüntüleme')
                    .setDescription(`Oyuncu ID **${oyuncuId}**'nin envanteri`)
                    .addFields({
                        name: '📋 Envanter İçeriği',
                        value: tableContent.length > 1024 
                            ? tableContent.substring(0, 1020) + '...```' 
                            : tableContent,
                        inline: false
                    })
                    .addFields({
                        name: '👤 Oyuncu ID',
                        value: oyuncuId.toString(),
                        inline: true
                    }, {
                        name: '📦 Toplam Item',
                        value: inventory.length.toString(),
                        inline: true
                    })
                    .setFooter({ text: `RoveBot's • Envanter Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [embed] });

                // Log gönder
                const inventoryList = inventory.map(item => 
                    `${item.label || item.name} (x${item.count || 1})`
                ).join(', ');

                await sendLog(interaction.client, interaction.guild.id, 'envantergoruntule', {
                    color: 0x00ff00,
                    title: '📦 Envanter Görüntüleme Logu',
                    description: `Bir oyuncunun envanteri görüntülendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '📦 Toplam Item',
                            value: inventory.length.toString(),
                            inline: true
                        },
                        {
                            name: '📋 Envanter İçeriği',
                            value: inventoryList.length > 1024 
                                ? inventoryList.substring(0, 1020) + '...' 
                                : inventoryList || 'Boş',
                            inline: false
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Envanter görüntülenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'envantergoruntule', {
                    color: 0xff0000,
                    title: '❌ Envanter Görüntüleme Hatası',
                    description: `Envanter görüntüleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Envanter-goruntule komutunda hata:', error);
            
            let errorMessage = 'Envanter görüntülenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'envantergoruntule', {
                color: 0xff0000,
                title: '❌ Envanter Görüntüleme Sistem Hatası',
                description: `Envanter görüntüleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

