const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('item-sil')
        .setDescription('Oyuncudan item siler')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1))
        .addStringOption(option =>
            option.setName('item')
                .setDescription('Item adı (örn: bread, water)')
                .setRequired(true))
        .addIntegerOption(option =>
            option.setName('miktar')
                .setDescription('Silinecek item miktarı (varsayılan: 1)')
                .setRequired(false)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'itemsil')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');
            const item = interaction.options.getString('item');
            const miktar = interaction.options.getInteger('miktar') || 1;

            // API isteği için body hazırla
            const requestBody = {
                playerId: oyuncuId,
                item: item,
                itemCount: miktar
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/removeitem`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Item Silindi')
                    .setDescription(`Oyuncu ID **${oyuncuId}**'den item başarıyla silindi!`)
                    .addFields(
                        {
                            name: '📦 Item',
                            value: item,
                            inline: true
                        },
                        {
                            name: '🔢 Silinen Miktar',
                            value: miktar.toString(),
                            inline: true
                        },
                        {
                            name: '👤 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Item Silme Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'itemsil', {
                    color: 0x00ff00,
                    title: '🗑️ Item Silme Logu',
                    description: `Bir oyuncudan item silindi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '📦 Item',
                            value: item,
                            inline: true
                        },
                        {
                            name: '🔢 Silinen Miktar',
                            value: miktar.toString(),
                            inline: true
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Item silinirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'itemsil', {
                    color: 0xff0000,
                    title: '❌ Item Silme Hatası',
                    description: `Item silme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '📦 Item',
                            value: item,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Item-sil komutunda hata:', error);
            
            let errorMessage = 'Item silinirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'itemsil', {
                color: 0xff0000,
                title: '❌ Item Silme Sistem Hatası',
                description: `Item silme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

