const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('kill')
        .setDescription('Oyuncuyu öldürür')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'kill')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');

            // API isteği için body hazırla
            const requestBody = {
                playerId: oyuncuId
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/kill`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const successEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('💀 Oyuncu Öldürüldü')
                    .setDescription(`Oyuncu ID **${oyuncuId}** başarıyla öldürüldü!`)
                    .addFields({
                        name: '👤 Oyuncu ID',
                        value: oyuncuId.toString(),
                        inline: true
                    })
                    .setFooter({ text: `RoveBot's • Kill Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'kill', {
                    color: 0xff0000,
                    title: '💀 Kill Logu',
                    description: `Bir oyuncu öldürüldü`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Oyuncu öldürülürken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'kill', {
                    color: 0xff0000,
                    title: '❌ Kill Hatası',
                    description: `Kill işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Kill komutunda hata:', error);
            
            let errorMessage = 'Oyuncu öldürülürken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'kill', {
                color: 0xff0000,
                title: '❌ Kill Sistem Hatası',
                description: `Kill işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

