const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, ChannelType, PermissionOverwrites } = require('discord.js');
const fs = require('fs');
const path = require('path');

// Log kanalları tanımları
const logChannels = {
    'aracver': '🚗 ᴀʀᴀᴄ ᴠᴇʀᴍᴇ ʟᴏɢ',
    'ban': '🚫 ʙᴀɴ ʟᴏɢ',
    'unban': '✅ ᴜɴʙᴀɴ ʟᴏɢ',
    'banbilgi': '📋 ʙᴀɴ ʙɪʟɢɪ ʟᴏɢ',
    'envantergoruntule': '🎒 ᴇɴᴠᴀɴᴛᴇʀ ɢᴏʀᴜɴᴛᴜʟᴇ ʟᴏɢ',
    'itemver': '📦 ɪᴛᴇᴍ ᴠᴇʀᴍᴇ ʟᴏɢ',
    'itemsil': '🗑️ ɪᴛᴇᴍ ꜱɪʟ ʟᴏɢ',
    'meslekver': '💼 ᴍᴇꜱʟᴇᴋ ᴠᴇʀᴍᴇ ʟᴏɢ',
    'oyuncubilgi': '👤 ᴏʏᴜɴᴄᴜ ʙɪʟɢɪ ʟᴏɢ',
    'oyunculistele': '📋 ᴏʏᴜɴᴄᴜ ʟɪꜱᴛᴇʟᴇ ʟᴏɢ',
    'paraver': '💰 ᴘᴀʀᴀ ᴠᴇʀᴍᴇ ʟᴏɢ',
    'parasil': '💸 ᴘᴀʀᴀ ꜱɪʟ ʟᴏɢ',
    'paraayarla': '⚙️ ᴘᴀʀᴀ ᴀʏᴀʀʟᴀ ʟᴏɢ',
    'revive': '💉 ʀᴇᴠɪᴠᴇ ʟᴏɢ',
    'kill': '💀 ᴋɪʟʟ ʟᴏɢ',
    'whitelistver': '✅ ᴡʜɪᴛᴇʟɪꜱᴛ ᴠᴇʀᴍᴇ ʟᴏɢ',
    'whitelistsil': '🗑️ ᴡʜɪᴛᴇʟɪꜱᴛ ꜱɪʟ ʟᴏɢ',
    'whitelistbilgi': '📋 ᴡʜɪᴛᴇʟɪꜱᴛ ʙɪʟɢɪ ʟᴏɢ'
};

module.exports = {
    data: new SlashCommandBuilder()
        .setName('logsistemikur')
        .setDescription('Tüm log kanallarını otomatik oluşturur')
        .addRoleOption(option =>
            option.setName('rol')
                .setDescription('Log kanallarını görebilecek rol(ler) - Boş bırakılırsa herkes görebilir')
                .setRequired(false)),

    async execute(interaction) {
        try {
            // Sadece Administrator yetkisi kontrolü
            if (!interaction.member.permissions.has(PermissionFlagsBits.Administrator)) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için **Administrator** yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply({ ephemeral: true });

            const rol = interaction.options.getRole('rol');

            // Kategori oluştur
            const categoryName = 'ʀᴏᴠᴇ ʙᴏᴛ\'ꜱ ɪᴄ ʙᴏᴛ ʟᴏɢʟᴀʀɪ';
            let category = interaction.guild.channels.cache.find(
                c => c.type === ChannelType.GuildCategory && c.name === categoryName
            );

            if (!category) {
                category = await interaction.guild.channels.create({
                    name: categoryName,
                    type: ChannelType.GuildCategory,
                    permissionOverwrites: rol ? [
                        {
                            id: interaction.guild.roles.everyone.id,
                            deny: [PermissionFlagsBits.ViewChannel]
                        },
                        {
                            id: rol.id,
                            allow: [PermissionFlagsBits.ViewChannel, PermissionFlagsBits.ReadMessageHistory]
                        }
                    ] : []
                });
            }

            const logPath = path.join(__dirname, '../jsons/logKanallari.json');
            let logData = {};

            if (fs.existsSync(logPath)) {
                try {
                    const fileContent = fs.readFileSync(logPath, 'utf8');
                    logData = fileContent.trim() ? JSON.parse(fileContent) : {};
                } catch (error) {
                    console.error('Log kanalları JSON parse hatası:', error);
                    logData = {};
                }
            }

            if (!logData[interaction.guild.id]) {
                logData[interaction.guild.id] = {};
            }

            const guildLogData = logData[interaction.guild.id];
            const createdChannels = [];
            const existingChannels = [];

            // Kanal isimleri mapping
            const channelNameMap = {
                'aracver': '🚗・ᴀʀᴀᴄ ᴠᴇʀᴍᴇ ʟᴏɢ',
                'ban': '🚫・ʙᴀɴ ʟᴏɢ',
                'unban': '✅・ᴜɴʙᴀɴ ʟᴏɢ',
                'banbilgi': '📋・ʙᴀɴ ʙɪʟɢɪ ʟᴏɢ',
                'envantergoruntule': '🎒・ᴇɴᴠᴀɴᴛᴇʀ ɢᴏʀᴜɴᴛᴜʟᴇ ʟᴏɢ',
                'itemver': '📦・ɪᴛᴇᴍ ᴠᴇʀᴍᴇ ʟᴏɢ',
                'itemsil': '🗑️・ɪᴛᴇᴍ ꜱɪʟ ʟᴏɢ',
                'meslekver': '💼・ᴍᴇꜱʟᴇᴋ ᴠᴇʀᴍᴇ ʟᴏɢ',
                'oyuncubilgi': '👤・ᴏʏᴜɴᴄᴜ ʙɪʟɢɪ ʟᴏɢ',
                'oyunculistele': '📋・ᴏʏᴜɴᴄᴜ ʟɪꜱᴛᴇʟᴇ ʟᴏɢ',
                'paraver': '💰・ᴘᴀʀᴀ ᴠᴇʀᴍᴇ ʟᴏɢ',
                'parasil': '💸・ᴘᴀʀᴀ ꜱɪʟ ʟᴏɢ',
                'paraayarla': '⚙️・ᴘᴀʀᴀ ᴀʏᴀʀʟᴀ ʟᴏɢ',
                'revive': '💉・ʀᴇᴠɪᴠᴇ ʟᴏɢ',
                'kill': '💀・ᴋɪʟʟ ʟᴏɢ',
                'whitelistver': '✅・ᴡʜɪᴛᴇʟɪꜱᴛ ᴠᴇʀᴍᴇ ʟᴏɢ',
                'whitelistsil': '🗑️・ᴡʜɪᴛᴇʟɪꜱᴛ ꜱɪʟ ʟᴏɢ',
                'whitelistbilgi': '📋・ᴡʜɪᴛᴇʟɪꜱᴛ ʙɪʟɢɪ ʟᴏɢ'
            };

            // Her log kanalı için kontrol et ve eksik olanları oluştur
            for (const [logKey, logName] of Object.entries(logChannels)) {
                try {
                    const channelName = channelNameMap[logKey] || logKey.toLowerCase();
                    let channel = null;

                    // Önce JSON'da kayıtlı mı kontrol et
                    if (guildLogData[logKey] && guildLogData[logKey].channelId) {
                        const savedChannelId = guildLogData[logKey].channelId;
                        // Discord'da bu kanal hala var mı kontrol et
                        channel = interaction.guild.channels.cache.get(savedChannelId);
                        
                        if (channel && channel.parentId === category.id) {
                            // Kanal mevcut ve doğru kategoride
                            existingChannels.push({ key: logKey, name: logName, channel });
                            continue; // Bu kanal zaten var, atla
                        }
                    }

                    // JSON'da yoksa veya Discord'da silinmişse, Discord'da ara
                    if (!channel) {
                        channel = interaction.guild.channels.cache.find(
                            c => c.name === channelName && c.parentId === category.id && c.type === ChannelType.GuildText
                        );
                    }

                    if (!channel) {
                        // Kanal yok, oluştur
                        channel = await interaction.guild.channels.create({
                            name: channelName,
                            type: ChannelType.GuildText,
                            parent: category.id,
                            permissionOverwrites: rol ? [
                                {
                                    id: interaction.guild.roles.everyone.id,
                                    deny: [PermissionFlagsBits.ViewChannel]
                                },
                                {
                                    id: rol.id,
                                    allow: [PermissionFlagsBits.ViewChannel, PermissionFlagsBits.ReadMessageHistory]
                                }
                            ] : []
                        });

                        createdChannels.push({ key: logKey, name: logName, channel });
                    } else {
                        // Kanal var ama JSON'da kayıtlı değil, ekle
                        existingChannels.push({ key: logKey, name: logName, channel });
                    }

                    // JSON'a kaydet/güncelle
                    guildLogData[logKey] = {
                        channelId: channel.id,
                        roleId: rol ? rol.id : (guildLogData[logKey]?.roleId || null),
                        logType: logKey
                    };

                } catch (error) {
                    console.error(`${logKey} kanalı oluşturulurken hata:`, error);
                }
            }

            // JSON dosyasını güncelle
            fs.writeFileSync(logPath, JSON.stringify(logData, null, 2));

            // Başarı embed'i oluştur
            let successEmbed;

            // Eğer tüm kanallar zaten varsa ve yeni kanal oluşturulmadıysa
            if (createdChannels.length === 0 && existingChannels.length === Object.keys(logChannels).length) {
                successEmbed = new EmbedBuilder()
                    .setColor(0x3498db)
                    .setTitle('ℹ️ Log Sistemi Zaten Kurulu')
                    .setDescription(`Tüm log kanalları zaten mevcut! Eksik kanal bulunamadı.`)
                    .addFields(
                        {
                            name: '📁 Kategori',
                            value: `<#${category.id}>`,
                            inline: true
                        },
                        {
                            name: '📊 Toplam Kanal',
                            value: `${existingChannels.length} kanal mevcut`,
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Log Sistemi` })
                    .setTimestamp();
            } else {
                successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Log Sistemi Güncellendi')
                    .setDescription(createdChannels.length > 0 
                        ? `${createdChannels.length} eksik kanal oluşturuldu!` 
                        : 'Log sistemi kontrol edildi.')
                    .addFields(
                        {
                            name: '📁 Kategori',
                            value: `<#${category.id}>`,
                            inline: true
                        },
                        {
                            name: '👥 Görüntüleme Rolü',
                            value: rol ? `<@&${rol.id}>` : 'Herkes görebilir',
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Log Sistemi` })
                    .setTimestamp();

                if (createdChannels.length > 0) {
                    successEmbed.addFields({
                        name: '✅ Oluşturulan Kanallar',
                        value: createdChannels.map(c => `• ${c.name} (<#${c.channel.id}>)`).join('\n'),
                        inline: false
                    });
                }

                if (existingChannels.length > 0) {
                    successEmbed.addFields({
                        name: 'ℹ️ Mevcut Kanallar',
                        value: `${existingChannels.length} kanal zaten mevcut`,
                        inline: false
                    });
                }
            }

            await interaction.editReply({ embeds: [successEmbed] });

        } catch (error) {
            console.error('Logsistemikur komutunda hata:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Log kanalları oluşturulurken bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};

