const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('oyuncu-listele')
        .setDescription('Veritabanındaki tüm oyuncuları listeler')
        .addIntegerOption(option =>
            option.setName('sayfa')
                .setDescription('Sayfa numarası (varsayılan: 1)')
                .setRequired(false)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'oyunculistele')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const sayfa = interaction.options.getInteger('sayfa') || 1;
            const itemsPerPage = 20;

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/players/db/list`;
            const response = await axios.get(apiUrl, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`
                }
            });

            if (response.data.success) {
                const players = response.data.players || [];
                const totalCount = response.data.count || players.length;

                if (players.length === 0) {
                    const emptyEmbed = new EmbedBuilder()
                        .setColor(0xffa500)
                        .setTitle('📋 Oyuncu Listesi')
                        .setDescription('Veritabanında oyuncu bulunamadı!')
                        .setFooter({ text: `RoveBot's • Oyuncu Listeleme Sistemi` })
                        .setTimestamp();

                    await interaction.editReply({ embeds: [emptyEmbed] });
                    return;
                }

                // Sayfalama hesapla
                const totalPages = Math.ceil(players.length / itemsPerPage);
                const currentPage = Math.min(sayfa, totalPages);
                const startIndex = (currentPage - 1) * itemsPerPage;
                const endIndex = startIndex + itemsPerPage;
                const pagePlayers = players.slice(startIndex, endIndex);

                // Table formatında göster
                let tableContent = '```\n';
                tableContent += 'CitizenID'.padEnd(15) + '| İsim\n';
                tableContent += '-'.repeat(15) + '|' + '-'.repeat(30) + '\n';
                
                pagePlayers.forEach(player => {
                    const citizenId = (player.citizenid || 'Bilinmiyor').substring(0, 14);
                    const name = (player.name || 'Bilinmiyor').substring(0, 29);
                    tableContent += citizenId.padEnd(15) + '| ' + name + '\n';
                });
                
                tableContent += '```';

                const embed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('📋 Oyuncu Listesi')
                    .setDescription(`Veritabanındaki oyuncular (Toplam: ${totalCount})`)
                    .addFields({
                        name: '📋 Oyuncular',
                        value: tableContent.length > 1024 
                            ? tableContent.substring(0, 1020) + '...```' 
                            : tableContent,
                        inline: false
                    })
                    .addFields({
                        name: '📊 Sayfa',
                        value: `${currentPage}/${totalPages}`,
                        inline: true
                    }, {
                        name: '👥 Toplam',
                        value: totalCount.toString(),
                        inline: true
                    })
                    .setFooter({ text: `RoveBot's • Oyuncu Listeleme Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [embed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'oyunculistele', {
                    color: 0x00ff00,
                    title: '📋 Oyuncu Listeleme Logu',
                    description: `Veritabanındaki oyuncular listelendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '👥 Toplam Oyuncu',
                            value: totalCount.toString(),
                            inline: true
                        },
                        {
                            name: '📊 Sayfa',
                            value: `${currentPage}/${totalPages}`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Oyuncu listesi görüntülenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'oyunculistele', {
                    color: 0xff0000,
                    title: '❌ Oyuncu Listeleme Hatası',
                    description: `Oyuncu listeleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Oyuncu-listele komutunda hata:', error);
            
            let errorMessage = 'Oyuncu listesi görüntülenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'oyunculistele', {
                color: 0xff0000,
                title: '❌ Oyuncu Listeleme Sistem Hatası',
                description: `Oyuncu listeleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

