const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('para-sil')
        .setDescription('Oyuncudan para siler')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1))
        .addStringOption(option =>
            option.setName('tip')
                .setDescription('Para tipi')
                .setRequired(true)
                .addChoices(
                    { name: 'Nakit', value: 'cash' },
                    { name: 'Banka', value: 'bank' }
                ))
        .addIntegerOption(option =>
            option.setName('miktar')
                .setDescription('Silinecek para miktarı')
                .setRequired(true)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'parasil')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');
            const tip = interaction.options.getString('tip');
            const miktar = interaction.options.getInteger('miktar');

            // API isteği için body hazırla
            const requestBody = {
                playerId: oyuncuId,
                moneyType: tip,
                amount: miktar
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/removemoney`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const tipLabel = tip === 'cash' ? 'Nakit' : 'Banka';
                const successEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('💸 Para Silindi')
                    .setDescription(`Oyuncu ID **${oyuncuId}**'den ${tipLabel} parası başarıyla silindi!`)
                    .addFields(
                        {
                            name: '💰 Para Tipi',
                            value: tipLabel,
                            inline: true
                        },
                        {
                            name: '💵 Silinen Miktar',
                            value: `$${miktar.toLocaleString()}`,
                            inline: true
                        },
                        {
                            name: '👤 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Para Silme Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'parasil', {
                    color: 0xff0000,
                    title: '💸 Para Silme Logu',
                    description: `Bir oyuncudan para silindi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '💰 Para Tipi',
                            value: tipLabel,
                            inline: true
                        },
                        {
                            name: '💵 Silinen Miktar',
                            value: `$${miktar.toLocaleString()}`,
                            inline: true
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Para silinirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'parasil', {
                    color: 0xff0000,
                    title: '❌ Para Silme Hatası',
                    description: `Para silme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '💰 Para Tipi',
                            value: tip === 'cash' ? 'Nakit' : 'Banka',
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Para-sil komutunda hata:', error);
            
            let errorMessage = 'Para silinirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'parasil', {
                color: 0xff0000,
                title: '❌ Para Silme Sistem Hatası',
                description: `Para silme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

