const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('whitelist-bilgi')
        .setDescription('Discord ID\'nin whitelist bilgisini görüntüler')
        .addStringOption(option =>
            option.setName('discordid')
                .setDescription('Whitelist bilgisi görüntülenecek Discord ID')
                .setRequired(true)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'whitelistbilgi')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const discordId = interaction.options.getString('discordid');

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/whitelistinfo`;
            const response = await axios.get(apiUrl, {
                params: {
                    discordId: discordId
                },
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`
                }
            });

            if (response.data.success) {
                const whitelistInfo = response.data.whitelistInfo;
                
                const embed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Whitelist Bilgisi')
                    .setDescription(`Discord ID **${discordId}** whitelist bilgisi`)
                    .addFields(
                        {
                            name: '🆔 Discord ID',
                            value: whitelistInfo.discordId,
                            inline: true
                        },
                        {
                            name: '👤 Ekleyen',
                            value: whitelistInfo.addedBy || 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '📅 Eklenme Tarihi',
                            value: whitelistInfo.addedAt 
                                ? `<t:${Math.floor(new Date(whitelistInfo.addedAt).getTime() / 1000)}:F>`
                                : 'Bilinmiyor',
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Whitelist Bilgi Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [embed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'whitelistbilgi', {
                    color: 0x00ff00,
                    title: '📋 Whitelist Bilgi Logu',
                    description: `Bir Discord ID'nin whitelist bilgisi görüntülendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '👤 Ekleyen',
                            value: whitelistInfo.addedBy || 'Bilinmiyor',
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Whitelist bilgisi görüntülenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'whitelistbilgi', {
                    color: 0xff0000,
                    title: '❌ Whitelist Bilgi Hatası',
                    description: `Whitelist bilgisi görüntüleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Whitelist-bilgi komutunda hata:', error);
            
            let errorMessage = 'Whitelist bilgisi görüntülenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'whitelistbilgi', {
                color: 0xff0000,
                title: '❌ Whitelist Bilgi Sistem Hatası',
                description: `Whitelist bilgisi görüntüleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

