const express = require('express');

class KeepAlive {
    constructor(port = 9676) {
        this.app = express();
        this.port = port;
        this.server = null;
        this.memoryThreshold = 500;
        this.setupRoutes();
        this.setupMemoryMonitoring();
    }

    setupRoutes() {
        this.app.disable('x-powered-by');
        
        this.app.use((req, res, next) => {
            req.setTimeout(30000);
            res.setTimeout(30000);
            next();
        });

        this.app.get('/', (req, res) => {
            res.status(200).json({
                status: 'alive',
                message: 'Guard Botu çalışıyor!',
                timestamp: new Date().toISOString(),
                uptime: process.uptime(),
                memory: this.getMemoryUsage()
            });
        });

        this.app.get('/ping', (req, res) => {
            res.status(200).json({
                status: 'pong',
                timestamp: new Date().toISOString()
            });
        });

        this.app.get('/health', (req, res) => {
            const memUsage = process.memoryUsage();
            const uptime = process.uptime();
            
            const memoryMB = {
                rss: Math.round(memUsage.rss / 1024 / 1024),
                heapUsed: Math.round(memUsage.heapUsed / 1024 / 1024),
                heapTotal: Math.round(memUsage.heapTotal / 1024 / 1024),
                external: Math.round(memUsage.external / 1024 / 1024)
            };
            
            const isCritical = memoryMB.heapUsed > this.memoryThreshold;
            const isWarning = memoryMB.heapUsed > (this.memoryThreshold * 0.8);
            
            const healthStatus = {
                status: isCritical ? 'critical' : isWarning ? 'warning' : 'healthy',
                uptime: Math.floor(uptime),
                memory: memoryMB,
                memoryThreshold: this.memoryThreshold,
                timestamp: new Date().toISOString(),
                pid: process.pid,
                nodeVersion: process.version,
                platform: process.platform
            };
            
            const statusCode = isCritical ? 503 : isWarning ? 200 : 200;
            
            res.status(statusCode).json(healthStatus);
        });

        this.app.get('/gc', (req, res) => {
            if (global.gc) {
                global.gc();
                res.status(200).json({
                    status: 'garbage collection triggered',
                    memory: this.getMemoryUsage(),
                    timestamp: new Date().toISOString()
                });
            } else {
                res.status(503).json({
                    error: 'Garbage collection not available. Start with --expose-gc flag.',
                    timestamp: new Date().toISOString()
                });
            }
        });

        this.app.use((req, res) => {
            res.status(404).json({
                error: 'Endpoint bulunamadı',
                availableEndpoints: ['/', '/ping', '/health', '/gc']
            });
        });
    }

    getMemoryUsage() {
        const memoryUsage = process.memoryUsage();
        return {
            rss: `${Math.round(memoryUsage.rss / 1024 / 1024)} MB`,
            heapTotal: `${Math.round(memoryUsage.heapTotal / 1024 / 1024)} MB`,
            heapUsed: `${Math.round(memoryUsage.heapUsed / 1024 / 1024)} MB`,
            external: `${Math.round(memoryUsage.external / 1024 / 1024)} MB`
        };
    }

    setupMemoryMonitoring() {
        setInterval(() => {
            const memoryUsage = process.memoryUsage();
            const heapUsedMB = memoryUsage.heapUsed / 1024 / 1024;
            
            if (heapUsedMB > this.memoryThreshold) {
                console.warn(`⚠️ Yüksek bellek kullanımı tespit edildi: ${Math.round(heapUsedMB)} MB`);
                
                if (global.gc) {
                    console.log('🗑️ Garbage collection tetikleniyor...');
                    global.gc();
                    
                    const newMemoryUsage = process.memoryUsage();
                    const newHeapUsedMB = newMemoryUsage.heapUsed / 1024 / 1024;
                    console.log(`📊 GC sonrası bellek kullanımı: ${Math.round(newHeapUsedMB)} MB`);
                }
            }
        }, 5 * 60 * 1000);
    }

    start() {
        return new Promise((resolve, reject) => {
            this.server = this.app.listen(this.port, (err) => {
                if (err) {
                    console.error('❌ Keep Alive servisi başlatılamadı:', err);
                    reject(err);
                } else {
                    console.log(`🌐 Keep Alive servisi başlatıldı: http://localhost:${this.port}`);
                    console.log(`📡 Health Check: http://localhost:${this.port}/health`);
                    resolve(this.server);
                }
            });
        });
    }

    stop() {
        return new Promise((resolve) => {
            if (this.server) {
                this.server.close(() => {
                    console.log('🔴 Keep Alive servisi durduruldu.');
                    resolve();
                });
            } else {
                resolve();
            }
        });
    }

    setupGracefulShutdown() {
        const gracefulShutdown = async (signal) => {
            console.log(`\n📡 ${signal} sinyali alındı. Keep Alive servisi kapatılıyor...`);
            await this.stop();
            process.exit(0);
        };

        process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
        process.on('SIGINT', () => gracefulShutdown('SIGINT'));
    }
}

module.exports = KeepAlive;